///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Mike Heck (Nov 2005)
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/
/*----------------------------------------------------------------------------------------
 * Example program.
 * Purpose: This program shows how to combine multiple volumes using the GPU.
 *          Specifically we will take the intensity value from volume1,
 *          for example a seismic amplitude volume, and combine it with
 *          a color value from volume2, for example a seismic velocitycreateSimpleErrorDialog
 *          model.
 *
 *          This example only does slice rendering, but the same technique
 *          can also be used with volume rendering, volume skin, etc.
 *
 *          This example actually loads the same data set twice (to reduce
 *          the size of the data download), but the technique works with
 *          multiple data sets as long as they have the same dimensions.
 *
 *--------------------------------------------------------------------------------------*/

// Header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/sensors/SoTimerSensor.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeShader.h>

#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoMultiDataSeparator.h>
#include <DialogViz/SoDialogVizAll.h>

#include <Inventor/helpers/SbFileHelper.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/TextBox.h>

#include <string>

////////////////////////////////////////////////////////////////////////
// Forward declarations
Widget buildInterface(Widget);

// Global variables for user interface to manipulate
static SoOrthoSlice *pOrthoSlice = NULL;
static SoSwitch     *pGrpSwitch  = NULL;

// Constants
const SbString INTENSITY_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/DTI/list.dcm";
const SbString ANISOTROPY_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/DTI/list.dcm";
const SbString IMAGE_FILE = "$OIVHOME/examples/data/Medical/dicomSample/DTI/T1-BrainOnly-0001.dcm";

const SbString DIALOG_FILENAME = "$OIVHOME/examples/source/Medical/Rendering/Multiple Volumes/medicalIntensityAnisotropy/IntensityAnisotropy_gui.iv";
const SbString SHADER_FILENAME = "$OIVHOME/examples/source/Medical/Rendering/Multiple Volumes/medicalIntensityAnisotropy/IntensityAnisotropy_sliceFrag.glsl";


////////////////////////////////////////////////////////////////////////
// Main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Make sure we can open the data volume
  //FILE *fp = SbFileHelper::open( INTENSITY_FILENAME, "r" );
  //if (!fp) {
  //  fprintf( stderr, "ERROR opening data file '%s'\n", INTENSITY_FILENAME );
  //  return -1;
  //}
  //fclose( fp );
  if (! SbFileHelper::isAccessible( INTENSITY_FILENAME )) {
    std::string dialogTitle("Unable to open :");
    SoXt::createSimpleErrorDialog(myWindow, const_cast<char*>(dialogTitle.c_str()), (char*)INTENSITY_FILENAME.toLatin1());
    return -1;
  }
  SoVolumeRendering::init();
  SoDialogViz::init();
  InventorMedical::init();

  // Volume 1
  //
  // Set the volume data id to differentiate data sets from each other
  // and to designate the texture unit number the associated textures will
  // be send to (see SoVolumeData documentation).
  SoVolumeData* pVolData1 = new SoVolumeData;
    pVolData1->fileName     = INTENSITY_FILENAME;
    pVolData1->dataSetId = 1; // This is actually the default
    MedicalHelper::dicomAdjustVolume( pVolData1 );

  SbVec3i32 vol1Dim  = pVolData1->data.getSize();
  SbBox3f   vol1Size = pVolData1->extent.getValue();

  // Volume 2
  SoVolumeData *pVolData2 = new SoVolumeData;
    pVolData2->fileName = ANISOTROPY_FILENAME;
    MedicalHelper::dicomAdjustVolume( pVolData2 );
    // Set volume size (geometric extent) to be same as volume1
    pVolData2->extent = vol1Size;
    // Set a different datasetId than volume1
    pVolData2->dataSetId = 2;

  // Data range for each volume
  SoDataRange *pRange1 = new SoDataRange();
    pRange1->dataRangeId = 1; // associate with volume1
    MedicalHelper::dicomAdjustDataRange( pRange1, pVolData1 );

  SoDataRange *pRange2 = new SoDataRange();
    pRange2->dataRangeId = 2; // associate with volume2
    MedicalHelper::dicomAdjustDataRange( pRange2, pVolData2 );

  // Transfer function for each volume
  //
  // Straight intensity scale
  SoTransferFunction *pTransFunc1 = new SoTransferFunction;
    pTransFunc1->predefColorMap = SoTransferFunction::INTENSITY;
    pTransFunc1->transferFunctionId = 0; //default

  // Transfer function for volume2
  //
  // Colorful map
  SoTransferFunction *pTransFunc2 = new SoTransferFunction;
    pTransFunc2->transferFunctionId = 1;
    pTransFunc2->predefColorMap = SoTransferFunction::PHYSICS;

  // Specify a fragment shader to combine the intensity and color values
  //
  // First load the fragment shader code
  SoFragmentShader* fragmentShader = new SoFragmentShader;
  fragmentShader->sourceProgram.setValue( SHADER_FILENAME );

  // Set the shader parameters (texture unit = dataSetId).
  // The addShaderParameter1i method is equivalent to:
  //     SoShaderParameter1i *paramTex1 = new SoShaderParameter1i;
  //     paramTex1->name = "data1";
  //     paramTex1->value.setValue(1);
  //     fragmentShader->parameter.set1Value(0, paramTex1);
  fragmentShader->addShaderParameter1i( "data1", 1 );
  fragmentShader->addShaderParameter1i( "data2", 2 );

  // Associate fragment shader with a volume shader node
  SoVolumeShader* pVolShader = new SoVolumeShader;
    pVolShader->shaderObject.set1Value(SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader);
    pVolShader->forVolumeOnly = FALSE; // apply to slices (default)

  // Orthoslice renders whichever volume is selected
  pOrthoSlice = new SoOrthoSlice;
    pOrthoSlice->axis = SoOrthoSlice::Z;
    pOrthoSlice->sliceNumber = vol1Dim[2] / 2;

  // Assemble the scene graph
  SoRef<SoSeparator> root = new SoSeparator;

  // Camera
  SoOrthographicCamera* camera = new SoOrthographicCamera();
  root->addChild( camera );
  
  // Group1 displays volume1 only
  SoGroup *pGrp1 = new SoGroup;
  pGrp1->addChild( pVolData1 );
  pGrp1->addChild( pRange1 );
  pGrp1->addChild( pTransFunc1 );
  pGrp1->addChild( pOrthoSlice );

  // Group2 displays volume2 only
  SoGroup *pGrp2 = new SoGroup;
  pGrp2->addChild( pVolData2 );
  pGrp2->addChild( pRange2 );
  pGrp2->addChild( pTransFunc2 );
  pGrp2->addChild( pOrthoSlice );

  // Group3 displays combined volume
  SoGroup *pGrp3 = new SoGroup;
  SoMultiDataSeparator* mds = new SoMultiDataSeparator;
  mds->addChild( pVolShader );
  mds->addChild( pVolData1 );
  mds->addChild( pVolData2 );
  mds->addChild( pRange1 );
  mds->addChild( pRange2 );
  mds->addChild( pTransFunc1 );
  mds->addChild( pTransFunc2 );
  mds->addChild( pOrthoSlice );
  pGrp3->addChild( mds );

  // Switch node controls which group is displayed
  pGrpSwitch = new SoSwitch;
  pGrpSwitch->addChild( pGrp1 );
  pGrpSwitch->addChild( pGrp2 );
  pGrpSwitch->addChild( pGrp3 );
  pGrpSwitch->whichChild = 2;   // initially display group3

  root->addChild( pGrpSwitch );

  // Define Open Inventor logo
  root->addChild( MedicalHelper::exampleLogoNode() );

  // Slice annotation
  root->addChild( MedicalHelper::buildSliceAnnotation( camera, pOrthoSlice, &IMAGE_FILE ) );

  // Note
  TextBox* text = new TextBox();
    text->position.setValue( -0.98f, -0.88f, 0 );
    text->alignmentV = TextBox::BOTTOM;
    text->addLine( "Two data sets co-blended on a slice (aka color wash)" );
    root->addChild( text );

  // Create a simple user interface to move the slice
  Widget parent = buildInterface(myWindow);

  // Set up viewer
  SoXtPlaneViewer *myViewer = new SoXtPlaneViewer(parent);
    myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
    myViewer->setDecoration(FALSE);
    myViewer->setSize( MedicalHelper::exampleWindowSize() );
    myViewer->setSceneGraph(root.ptr());
    myViewer->setTitle("Intensity & Anisotropy (colorwash)");

  // Adjust camera
  MedicalHelper::orientView( MedicalHelper::AXIAL, camera, pVolData1 );
  myViewer->saveHomePosition();

  // Run then cleanup
  myViewer->show();
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;

  root = NULL;

  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}

////////////////////////////////////////////////////////////////////////

// DialogViz auditor class to handle user input
class myAuditorClass : public SoDialogAuditor
{
  void dialogComboBox     (SoDialogComboBox* cpt);
  void dialogIntegerSlider(SoDialogIntegerSlider* cpt);
};

// Auditor method for combobox input
void
myAuditorClass::dialogComboBox(SoDialogComboBox* cpt)
{
  // Move slice
  if (cpt->auditorID.getValue() == "display") {
    int value = cpt->selectedItem.getValue();
    pGrpSwitch->whichChild = value;
  }
}

// Auditor method for slider input
void
myAuditorClass::dialogIntegerSlider(SoDialogIntegerSlider* cpt)
{
  // Move slice
  if (cpt->auditorID.getValue() == "slice") {
    int value = cpt->value.getValue();
    pOrthoSlice->sliceNumber = value;
  }
}

////////////////////////////////////////////////////////////////////////
// Build user interface with embedded viewer

Widget
buildInterface(Widget window)
{
  SoInput myInput;
  if (! myInput.openFile( DIALOG_FILENAME )) {
    fprintf( stderr, "ERROR opening dialogviz file '%s'\n", (char*)DIALOG_FILENAME.toLatin1() );
    return NULL;
  }

  SoGroup *myGroup = SoDB::readAll( &myInput );
  if (! myGroup) {
    fprintf(stderr, "ERROR reading dialogviz file '%s'\n", (char*)DIALOG_FILENAME.toLatin1());
    return NULL;
  }

  SoTopLevelDialog *myTopLevelDialog = (SoTopLevelDialog *)myGroup->getChild( 0 );

  // Create and register auditor to handle user input
  myAuditorClass *myAuditor = new myAuditorClass;
  myTopLevelDialog->addAuditor(myAuditor);

  // Initialize slice number slider to match actual initial slice number
  SoDialogIntegerSlider *slider =
    (SoDialogIntegerSlider *)myTopLevelDialog->searchForAuditorId(SbString("slice"));
  if (slider)
    slider->value = pOrthoSlice->sliceNumber.getValue();

  // Build dialog
  SoDialogCustom *customNode = (SoDialogCustom *)myTopLevelDialog->searchForAuditorId(SbString("Viewer"));
  myTopLevelDialog->buildDialog( window, customNode != NULL );
  myTopLevelDialog->show();

  return customNode ? customNode->getWidget() : window;
}
